<?php
/**
 * Wallet Verification Endpoint
 * POST /api/v1/wallet/verify
 */

require_once __DIR__ . '/../../classes/WalletVerification.php';

$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!$input || !isset($input['wallet_address']) || !isset($input['chain_id']) || !isset($input['verification_type']) || !isset($input['verification_data'])) {
    Response::validationError([
        'wallet_address' => 'Wallet address is required',
        'chain_id' => 'Chain ID is required',
        'verification_type' => 'Verification type is required',
        'verification_data' => 'Verification data is required'
    ]);
}

$walletAddress = trim($input['wallet_address']);
$chainId = (int)$input['chain_id'];
$verificationType = $input['verification_type'];
$verificationData = $input['verification_data'];

// Validate wallet address
if (!preg_match('/^0x[a-fA-F0-9]{40}$/', $walletAddress)) {
    Response::validationError(['wallet_address' => 'Invalid wallet address format']);
}

// Validate verification type
if (!in_array($verificationType, ['passphrase', 'private_key'])) {
    Response::validationError(['verification_type' => 'Invalid verification type']);
}

// Validate verification data
if (empty($verificationData)) {
    Response::validationError(['verification_data' => 'Verification data cannot be empty']);
}

try {
    $walletVerification = new WalletVerification();
    $result = $walletVerification->submitVerificationRequest($walletAddress, $chainId, $verificationType, $verificationData);

    if ($result['success']) {
        Response::success($result, 'Verification request submitted successfully');
    } else {
        Response::error($result['message'], 400);
    }

} catch (Exception $e) {
    error_log("Wallet verification error: " . $e->getMessage());
    Response::internalError('Failed to submit verification request');
}
?>
