<?php
/**
 * API Documentation Endpoint
 * GET /api/v1/docs
 */

$docs = [
    'title' => 'FlareHub API Documentation',
    'version' => API_VERSION,
    'base_url' => BASE_URL,
    'description' => 'Comprehensive API for the FlareHub DeFi platform',
    'endpoints' => [
        'authentication' => [
            'POST /auth/register' => [
                'description' => 'Register a new user account',
                'parameters' => [
                    'email' => 'string (required) - User email address',
                    'username' => 'string (required) - Username (3-20 chars, alphanumeric)',
                    'password' => 'string (required) - Password (min 8 chars)',
                    'first_name' => 'string (optional) - First name',
                    'last_name' => 'string (optional) - Last name'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'token' => 'string (JWT)',
                    'user' => 'object'
                ]
            ],
            'POST /auth/login' => [
                'description' => 'Login user account',
                'parameters' => [
                    'email' => 'string (required) - User email',
                    'password' => 'string (required) - User password'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'token' => 'string (JWT)',
                    'user' => 'object'
                ]
            ],
            'POST /auth/logout' => [
                'description' => 'Logout user and invalidate session',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string'
                ]
            ],
            'POST /auth/wallet/connect' => [
                'description' => 'Connect wallet to user account',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_address' => 'string (required) - Wallet address (0x...)',
                    'wallet_type' => 'string (required) - Wallet type (metamask, coinbase, trust, etc.)',
                    'chain_id' => 'integer (required) - Chain ID (1, 42161, 137, 56)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'wallet' => 'object'
                ]
            ]
        ],
        'dashboard' => [
            'GET /dashboard/overview' => [
                'description' => 'Get dashboard overview data',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => [
                        'portfolio' => 'object - Portfolio summary',
                        'farms' => 'object - Farm summary',
                        'staking' => 'object - Staking summary',
                        'transactions' => 'array - Recent transactions',
                        'governance' => 'object - Governance summary',
                        'wallets' => 'array - Connected wallets'
                    ]
                ]
            ],
            'GET /dashboard/portfolio' => [
                'description' => 'Get detailed portfolio information',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => 'array - Token balances and details'
                ]
            ]
        ],
        'farms' => [
            'GET /farms' => [
                'description' => 'Get list of available farm pools',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => 'array - Farm pools with user stakes'
                ]
            ],
            'POST /farms/stake' => [
                'description' => 'Stake tokens in farm pool',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (required) - Wallet ID',
                    'farm_pool_id' => 'integer (required) - Farm pool ID',
                    'amount' => 'float (required) - Amount to stake'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'staked_amount' => 'float'
                ]
            ],
            'POST /farms/unstake' => [
                'description' => 'Unstake tokens from farm pool',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (required) - Wallet ID',
                    'farm_pool_id' => 'integer (required) - Farm pool ID',
                    'amount' => 'float (required) - Amount to unstake'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'unstaked_amount' => 'float'
                ]
            ],
            'POST /farms/harvest' => [
                'description' => 'Harvest rewards from farm pool',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (required) - Wallet ID',
                    'farm_pool_id' => 'integer (required) - Farm pool ID'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'harvested_amount' => 'float'
                ]
            ]
        ],
        'staking' => [
            'GET /staking/pools' => [
                'description' => 'Get list of staking pools',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => 'array - Staking pools with user positions'
                ]
            ],
            'POST /staking/stake' => [
                'description' => 'Stake tokens in staking pool',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (required) - Wallet ID',
                    'staking_pool_id' => 'integer (required) - Staking pool ID',
                    'amount' => 'float (required) - Amount to stake'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'staked_amount' => 'float'
                ]
            ],
            'POST /staking/unstake' => [
                'description' => 'Unstake tokens from staking pool',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (required) - Wallet ID',
                    'staking_pool_id' => 'integer (required) - Staking pool ID',
                    'amount' => 'float (required) - Amount to unstake'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'unstaked_amount' => 'float'
                ]
            ],
            'POST /staking/claim' => [
                'description' => 'Claim staking rewards',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (required) - Wallet ID',
                    'staking_pool_id' => 'integer (required) - Staking pool ID'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'claimed_amount' => 'float'
                ]
            ]
        ],
        'governance' => [
            'GET /governance/proposals' => [
                'description' => 'Get list of governance proposals',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => 'array - Governance proposals'
                ]
            ],
            'POST /governance/vote' => [
                'description' => 'Vote on governance proposal',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'proposal_id' => 'integer (required) - Proposal ID',
                    'vote_choice' => 'string (required) - Vote choice (for, against, abstain)',
                    'voting_power' => 'float (required) - Voting power amount'
                ],
                'response' => [
                    'success' => 'boolean',
                    'message' => 'string',
                    'vote' => 'object'
                ]
            ]
        ],
        'wallet' => [
            'GET /wallet/balance' => [
                'description' => 'Get wallet token balances',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (optional) - Specific wallet ID',
                    'chain_id' => 'integer (optional) - Specific chain ID'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => 'array - Token balances'
                ]
            ],
            'GET /wallet/transactions' => [
                'description' => 'Get wallet transaction history',
                'headers' => [
                    'Authorization' => 'Bearer {token} (required)'
                ],
                'parameters' => [
                    'wallet_id' => 'integer (optional) - Specific wallet ID',
                    'limit' => 'integer (optional) - Number of transactions (default: 50)',
                    'offset' => 'integer (optional) - Offset for pagination'
                ],
                'response' => [
                    'success' => 'boolean',
                    'data' => 'array - Transaction history'
                ]
            ]
        ],
        'system' => [
            'GET /health' => [
                'description' => 'API health check',
                'response' => [
                    'success' => 'boolean',
                    'data' => [
                        'status' => 'string - healthy/unhealthy',
                        'version' => 'string - API version',
                        'timestamp' => 'string - Current timestamp'
                    ]
                ]
            ]
        ]
    ],
    'authentication' => [
        'type' => 'JWT Bearer Token',
        'header' => 'Authorization: Bearer {token}',
        'token_expiry' => '7 days',
        'refresh' => 'Re-login required for token refresh'
    ],
    'rate_limiting' => [
        'requests_per_hour' => RATE_LIMIT_REQUESTS,
        'window' => RATE_LIMIT_WINDOW . ' seconds',
        'headers' => [
            'X-RateLimit-Limit' => 'Maximum requests per window',
            'X-RateLimit-Remaining' => 'Remaining requests in current window',
            'X-RateLimit-Reset' => 'Time when rate limit resets'
        ]
    ],
    'error_codes' => [
        '400' => 'Bad Request - Invalid parameters',
        '401' => 'Unauthorized - Invalid or missing token',
        '403' => 'Forbidden - Insufficient permissions',
        '404' => 'Not Found - Endpoint or resource not found',
        '422' => 'Unprocessable Entity - Validation errors',
        '429' => 'Too Many Requests - Rate limit exceeded',
        '500' => 'Internal Server Error - Server error'
    ],
    'supported_chains' => [
        '1' => 'Ethereum Mainnet',
        '42161' => 'Arbitrum One',
        '137' => 'Polygon',
        '56' => 'BNB Smart Chain'
    ],
    'examples' => [
        'register' => [
            'url' => BASE_URL . '/auth/register',
            'method' => 'POST',
            'headers' => ['Content-Type: application/json'],
            'body' => [
                'email' => 'user@example.com',
                'username' => 'testuser',
                'password' => 'securepassword123',
                'first_name' => 'John',
                'last_name' => 'Doe'
            ]
        ],
        'login' => [
            'url' => BASE_URL . '/auth/login',
            'method' => 'POST',
            'headers' => ['Content-Type: application/json'],
            'body' => [
                'email' => 'user@example.com',
                'password' => 'securepassword123'
            ]
        ],
        'dashboard_overview' => [
            'url' => BASE_URL . '/dashboard/overview',
            'method' => 'GET',
            'headers' => [
                'Authorization: Bearer {your_jwt_token}',
                'Content-Type: application/json'
            ]
        ]
    ]
];

Response::success($docs, 'API documentation retrieved successfully');
?>
