#!/bin/bash

# FlareHub Backend Deployment Script
# Run this script on your FlareHub.net server

echo "🚀 Starting FlareHub Backend Deployment..."

# Set variables
BACKUP_DIR="/var/backups/flarehub"
WEB_ROOT="/var/www/flarehub.net"
API_DIR="$WEB_ROOT/api"
DB_NAME="flarehub"
DB_USER="flarehub_user"

# Create backup directory
echo "📁 Creating backup directory..."
sudo mkdir -p $BACKUP_DIR

# Backup existing database (if exists)
echo "💾 Backing up existing database..."
if mysql -u $DB_USER -p -e "USE $DB_NAME;" 2>/dev/null; then
    mysqldump -u $DB_USER -p $DB_NAME > $BACKUP_DIR/flarehub_backup_$(date +%Y%m%d_%H%M%S).sql
    echo "✅ Database backup completed"
else
    echo "ℹ️  No existing database found, skipping backup"
fi

# Create web directory structure
echo "📂 Creating directory structure..."
sudo mkdir -p $API_DIR
sudo mkdir -p $API_DIR/logs
sudo mkdir -p $API_DIR/uploads

# Set permissions
echo "🔐 Setting permissions..."
sudo chown -R www-data:www-data $API_DIR
sudo chmod -R 755 $API_DIR
sudo chmod -R 777 $API_DIR/logs
sudo chmod -R 777 $API_DIR/uploads

# Copy backend files
echo "📋 Copying backend files..."
sudo cp -r backend/* $API_DIR/

# Install PHP dependencies (if using Composer)
if [ -f "$API_DIR/composer.json" ]; then
    echo "📦 Installing PHP dependencies..."
    cd $API_DIR
    sudo composer install --no-dev --optimize-autoloader
fi

# Create database and import schema
echo "🗄️  Setting up database..."
mysql -u root -p -e "CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
mysql -u root -p -e "CREATE USER IF NOT EXISTS '$DB_USER'@'localhost' IDENTIFIED BY 'your_secure_password_here';"
mysql -u root -p -e "GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USER'@'localhost';"
mysql -u root -p -e "FLUSH PRIVILEGES;"

# Import database schema
echo "📊 Importing database schema..."
mysql -u $DB_USER -p $DB_NAME < $API_DIR/database/schema.sql

# Configure Apache virtual host
echo "🌐 Configuring Apache virtual host..."
sudo tee /etc/apache2/sites-available/flarehub.net.conf > /dev/null <<EOF
<VirtualHost *:80>
    ServerName flarehub.net
    ServerAlias www.flarehub.net
    DocumentRoot $WEB_ROOT
    
    # API endpoints
    Alias /api $API_DIR
    <Directory $API_DIR>
        AllowOverride All
        Require all granted
    </Directory>
    
    # Security headers
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    
    # Logging
    ErrorLog \${APACHE_LOG_DIR}/flarehub_error.log
    CustomLog \${APACHE_LOG_DIR}/flarehub_access.log combined
</VirtualHost>

<VirtualHost *:443>
    ServerName flarehub.net
    ServerAlias www.flarehub.net
    DocumentRoot $WEB_ROOT
    
    # SSL Configuration
    SSLEngine on
    SSLCertificateFile /etc/ssl/certs/flarehub.net.crt
    SSLCertificateKeyFile /etc/ssl/private/flarehub.net.key
    
    # API endpoints
    Alias /api $API_DIR
    <Directory $API_DIR>
        AllowOverride All
        Require all granted
    </Directory>
    
    # Security headers
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
    
    # Logging
    ErrorLog \${APACHE_LOG_DIR}/flarehub_ssl_error.log
    CustomLog \${APACHE_LOG_DIR}/flarehub_ssl_access.log combined
</VirtualHost>
EOF

# Enable site and modules
echo "🔧 Enabling Apache modules and site..."
sudo a2enmod rewrite
sudo a2enmod headers
sudo a2enmod ssl
sudo a2enmod deflate
sudo a2enmod expires
sudo a2ensite flarehub.net.conf

# Restart Apache
echo "🔄 Restarting Apache..."
sudo systemctl restart apache2

# Set up SSL certificate (Let's Encrypt)
echo "🔒 Setting up SSL certificate..."
sudo apt-get update
sudo apt-get install -y certbot python3-certbot-apache
sudo certbot --apache -d flarehub.net -d www.flarehub.net --non-interactive --agree-tos --email admin@flarehub.net

# Create cron job for database cleanup
echo "⏰ Setting up cron jobs..."
(crontab -l 2>/dev/null; echo "0 2 * * * /usr/bin/mysql -u $DB_USER -p$DB_PASSWORD -e 'DELETE FROM api_rate_limits WHERE window_start < DATE_SUB(NOW(), INTERVAL 1 DAY);' $DB_NAME") | crontab -

# Create log rotation
echo "📝 Setting up log rotation..."
sudo tee /etc/logrotate.d/flarehub > /dev/null <<EOF
$API_DIR/logs/*.log {
    daily
    missingok
    rotate 30
    compress
    delaycompress
    notifempty
    create 644 www-data www-data
    postrotate
        /bin/systemctl reload apache2 > /dev/null 2>&1 || true
    endscript
}
EOF

# Test API endpoint
echo "🧪 Testing API endpoint..."
sleep 5
if curl -f -s "https://flarehub.net/api/v1/health" > /dev/null; then
    echo "✅ API is responding correctly"
else
    echo "❌ API test failed - check configuration"
fi

echo "🎉 FlareHub Backend deployment completed!"
echo "📋 Next steps:"
echo "   1. Update config/config.php with your actual API keys"
echo "   2. Update config/database.php with your database credentials"
echo "   3. Test all API endpoints"
echo "   4. Set up monitoring and alerts"
echo ""
echo "🔗 API Base URL: https://flarehub.net/api/v1"
echo "📚 API Documentation: https://flarehub.net/api/v1/docs"
