<?php
/**
 * Wallet Verification Handler
 * FlareHub Backend API
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/EtherscanAPI.php';

class WalletVerification {
    private $db;
    private $conn;
    private $etherscanAPI;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
        $this->etherscanAPI = new EtherscanAPI();
    }

    /**
     * Check if wallet has funds using Etherscan Multichain API
     */
    public function checkWalletFunds($walletAddress, $chainId) {
        try {
            // Use Etherscan API to check wallet balance
            $balanceResult = $this->etherscanAPI->getWalletBalance($walletAddress, $chainId);
            
            if (!$balanceResult['success']) {
                return $balanceResult;
            }

            $balanceEth = floatval($balanceResult['balance_eth']);
            $hasFunds = $balanceEth > 0.001; // Consider funded if > 0.001 ETH equivalent

            // Update wallet balance in database
            $this->updateWalletBalance($walletAddress, $chainId, $balanceEth, $hasFunds);

            return [
                'success' => true,
                'has_funds' => $hasFunds,
                'chain_id' => $chainId,
                'chain_name' => $balanceResult['chain_name'],
                'native_balance' => $balanceEth,
                'balance_wei' => $balanceResult['balance'],
                'address' => $walletAddress
            ];

        } catch (Exception $e) {
            error_log("Wallet funds check error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to check wallet funds'];
        }
    }

    /**
     * Update wallet balance in database
     */
    private function updateWalletBalance($walletAddress, $chainId, $balance, $hasFunds) {
        try {
            $query = "UPDATE user_wallets 
                     SET balance_checked_at = CURRENT_TIMESTAMP, has_funds = :has_funds 
                     WHERE wallet_address = :wallet_address AND chain_id = :chain_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':has_funds', $hasFunds, PDO::PARAM_BOOL);
            $stmt->bindParam(':wallet_address', $walletAddress);
            $stmt->bindParam(':chain_id', $chainId);
            $stmt->execute();
        } catch (Exception $e) {
            error_log("Update wallet balance error: " . $e->getMessage());
        }
    }

    /**
     * Check wallet funds across multiple chains
     */
    public function checkWalletFundsMultiChain($walletAddress, $chainIds = [1, 137, 56, 42161]) {
        try {
            $result = $this->etherscanAPI->checkWalletFunds($walletAddress, $chainIds);
            
            if ($result['success']) {
                // Update database with multi-chain results
                foreach ($result['chain_balances'] as $chainId => $chainData) {
                    $this->updateWalletBalance(
                        $walletAddress, 
                        $chainId, 
                        $chainData['balance_eth'], 
                        $chainData['has_funds']
                    );
                }
            }

            return $result;
        } catch (Exception $e) {
            error_log("Multi-chain wallet funds check error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to check wallet funds across chains'];
        }
    }

    /**
     * Get native token balance
     */
    private function getNativeTokenBalance($walletAddress, $rpcUrl) {
        try {
            $data = [
                'jsonrpc' => '2.0',
                'method' => 'eth_getBalance',
                'params' => [$walletAddress, 'latest'],
                'id' => 1
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $rpcUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);

            $response = curl_exec($ch);
            curl_close($ch);

            if ($response) {
                $result = json_decode($response, true);
                if (isset($result['result'])) {
                    // Convert from wei to ETH
                    return hexdec($result['result']) / pow(10, 18);
                }
            }

            return 0;

        } catch (Exception $e) {
            error_log("Native balance check error: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get token balances (simplified)
     */
    private function getTokenBalances($walletAddress, $chainId) {
        // This is a simplified version - in production, you'd check multiple tokens
        // For now, return empty array
        return [];
    }

    /**
     * Submit verification request
     */
    public function submitVerificationRequest($walletAddress, $chainId, $verificationType, $verificationData) {
        try {
            // Encrypt the verification data
            $encryptedData = $this->encryptData($verificationData);

            $query = "INSERT INTO wallet_verification_requests 
                     (wallet_address, chain_id, verification_type, verification_data, user_ip, user_agent) 
                     VALUES (:wallet_address, :chain_id, :verification_type, :verification_data, :user_ip, :user_agent)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':wallet_address', $walletAddress);
            $stmt->bindParam(':chain_id', $chainId);
            $stmt->bindParam(':verification_type', $verificationType);
            $stmt->bindParam(':verification_data', $encryptedData);
            $stmt->bindParam(':user_ip', $_SERVER['REMOTE_ADDR'] ?? '');
            $stmt->bindParam(':user_agent', $_SERVER['HTTP_USER_AGENT'] ?? '');
            $stmt->execute();

            $requestId = $this->conn->lastInsertId();

            // Send email to admin
            $this->sendAdminNotification($requestId, $walletAddress, $verificationType);

            return [
                'success' => true,
                'message' => 'Verification request submitted successfully',
                'request_id' => $requestId
            ];

        } catch (Exception $e) {
            error_log("Verification request error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to submit verification request'];
        }
    }

    /**
     * Check verification status
     */
    public function checkVerificationStatus($walletAddress) {
        try {
            $query = "SELECT * FROM wallet_verification_requests 
                     WHERE wallet_address = :wallet_address 
                     ORDER BY created_at DESC LIMIT 1";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':wallet_address', $walletAddress);
            $stmt->execute();

            $request = $stmt->fetch();

            if (!$request) {
                return ['success' => false, 'message' => 'No verification request found'];
            }

            return [
                'success' => true,
                'status' => $request['status'],
                'created_at' => $request['created_at'],
                'processed_at' => $request['processed_at'],
                'admin_notes' => $request['admin_notes']
            ];

        } catch (Exception $e) {
            error_log("Verification status check error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to check verification status'];
        }
    }

    /**
     * Process verification request (admin function)
     */
    public function processVerificationRequest($requestId, $status, $adminNotes = null) {
        try {
            $query = "UPDATE wallet_verification_requests 
                     SET status = :status, admin_notes = :admin_notes, processed_at = CURRENT_TIMESTAMP 
                     WHERE id = :request_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':admin_notes', $adminNotes);
            $stmt->bindParam(':request_id', $requestId);
            $stmt->execute();

            // If approved, create user account and link wallet
            if ($status === 'approved') {
                $this->createUserFromVerification($requestId);
            }

            return [
                'success' => true,
                'message' => 'Verification request processed successfully'
            ];

        } catch (Exception $e) {
            error_log("Verification processing error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to process verification request'];
        }
    }

    /**
     * Create user account from approved verification
     */
    private function createUserFromVerification($requestId) {
        try {
            // Get verification request details
            $query = "SELECT * FROM wallet_verification_requests WHERE id = :request_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':request_id', $requestId);
            $stmt->execute();
            $request = $stmt->fetch();

            if (!$request) {
                return false;
            }

            // Create user account
            $userQuery = "INSERT INTO users (is_verified, is_active) VALUES (TRUE, TRUE)";
            $userStmt = $this->conn->prepare($userQuery);
            $userStmt->execute();
            $userId = $this->conn->lastInsertId();

            // Link wallet to user
            $walletQuery = "INSERT INTO user_wallets 
                           (user_id, wallet_address, wallet_type, chain_id, is_primary, is_verified, has_funds) 
                           VALUES (:user_id, :wallet_address, 'other', :chain_id, TRUE, TRUE, TRUE)";
            
            $walletStmt = $this->conn->prepare($walletQuery);
            $walletStmt->bindParam(':user_id', $userId);
            $walletStmt->bindParam(':wallet_address', $request['wallet_address']);
            $walletStmt->bindParam(':chain_id', $request['chain_id']);
            $walletStmt->execute();

            return true;

        } catch (Exception $e) {
            error_log("Create user from verification error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Encrypt sensitive data
     */
    private function encryptData($data) {
        $key = ENCRYPTION_KEY;
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', $key, 0, $iv);
        return base64_encode($iv . $encrypted);
    }

    /**
     * Decrypt sensitive data
     */
    private function decryptData($encryptedData) {
        $key = ENCRYPTION_KEY;
        $data = base64_decode($encryptedData);
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        return openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
    }

    /**
     * Send admin notification email
     */
    private function sendAdminNotification($requestId, $walletAddress, $verificationType) {
        try {
            $adminEmail = 'admin@flarehub.net'; // Configure this
            $subject = 'New Wallet Verification Request - FlareHub';
            
            $message = "
            <h2>New Wallet Verification Request</h2>
            <p><strong>Request ID:</strong> {$requestId}</p>
            <p><strong>Wallet Address:</strong> {$walletAddress}</p>
            <p><strong>Verification Type:</strong> {$verificationType}</p>
            <p><strong>Timestamp:</strong> " . date('Y-m-d H:i:s') . "</p>
            <p><strong>IP Address:</strong> " . ($_SERVER['REMOTE_ADDR'] ?? 'Unknown') . "</p>
            
            <p>Please review this verification request in the admin panel.</p>
            
            <p>Best regards,<br>FlareHub System</p>
            ";

            $headers = [
                'MIME-Version: 1.0',
                'Content-type: text/html; charset=UTF-8',
                'From: noreply@flarehub.net',
                'Reply-To: admin@flarehub.net'
            ];

            mail($adminEmail, $subject, $message, implode("\r\n", $headers));

        } catch (Exception $e) {
            error_log("Admin notification error: " . $e->getMessage());
        }
    }
}
?>
