<?php
/**
 * JWT Token Handler
 * FlareHub Backend API
 */

class JWT {
    private static $secret;
    private static $algorithm;

    public static function init() {
        self::$secret = JWT_SECRET;
        self::$algorithm = JWT_ALGORITHM;
    }

    /**
     * Generate JWT token
     */
    public static function generate($payload) {
        self::init();
        
        $header = json_encode(['typ' => 'JWT', 'alg' => self::$algorithm]);
        $payload['iat'] = time();
        $payload['exp'] = time() + JWT_EXPIRY;
        $payload = json_encode($payload);
        
        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, self::$secret, true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        return $base64Header . "." . $base64Payload . "." . $base64Signature;
    }

    /**
     * Verify JWT token
     */
    public static function verify($token) {
        self::init();
        
        $tokenParts = explode('.', $token);
        if (count($tokenParts) !== 3) {
            return false;
        }
        
        $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[0]));
        $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[1]));
        $signature = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[2]));
        
        $expectedSignature = hash_hmac('sha256', $tokenParts[0] . "." . $tokenParts[1], self::$secret, true);
        
        if (!hash_equals($signature, $expectedSignature)) {
            return false;
        }
        
        $payloadData = json_decode($payload, true);
        
        if ($payloadData['exp'] < time()) {
            return false;
        }
        
        return $payloadData;
    }

    /**
     * Extract token from Authorization header
     */
    public static function extractToken() {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? '';
        
        if (preg_match('/Bearer\s(\S+)/', $authHeader, $matches)) {
            return $matches[1];
        }
        
        return null;
    }

    /**
     * Get current user from token
     */
    public static function getCurrentUser() {
        $token = self::extractToken();
        if (!$token) {
            return null;
        }
        
        $payload = self::verify($token);
        if (!$payload) {
            return null;
        }
        
        return $payload;
    }
}
?>
