<?php
/**
 * Dashboard Data Handler
 * FlareHub Backend API
 */

require_once __DIR__ . '/../config/database.php';

class Dashboard {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Get dashboard overview data
     */
    public function getOverview($userId) {
        try {
            // Get user wallets
            $wallets = $this->getUserWallets($userId);
            
            // Get portfolio summary
            $portfolio = $this->getPortfolioSummary($userId);
            
            // Get farm summary
            $farms = $this->getFarmSummary($userId);
            
            // Get staking summary
            $staking = $this->getStakingSummary($userId);
            
            // Get recent transactions
            $transactions = $this->getRecentTransactions($userId, 10);
            
            // Get governance summary
            $governance = $this->getGovernanceSummary($userId);

            return [
                'portfolio' => $portfolio,
                'farms' => $farms,
                'staking' => $staking,
                'transactions' => $transactions,
                'governance' => $governance,
                'wallets' => $wallets
            ];

        } catch (Exception $e) {
            error_log("Dashboard overview error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get user wallets
     */
    private function getUserWallets($userId) {
        $query = "SELECT uw.*, sc.name as chain_name, sc.symbol as chain_symbol 
                 FROM user_wallets uw 
                 JOIN supported_chains sc ON uw.chain_id = sc.chain_id 
                 WHERE uw.user_id = :user_id 
                 ORDER BY uw.is_primary DESC, uw.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    /**
     * Get portfolio summary
     */
    private function getPortfolioSummary($userId) {
        // Get total portfolio value (simplified - in production, fetch real-time prices)
        $query = "SELECT 
                    SUM(utb.balance * 1) as total_value, -- Simplified pricing
                    COUNT(DISTINCT utb.token_id) as token_count,
                    COUNT(DISTINCT utb.wallet_id) as wallet_count
                 FROM user_token_balances utb 
                 WHERE utb.user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();

        $result = $stmt->fetch();
        
        return [
            'total_value' => (float)($result['total_value'] ?? 0),
            'token_count' => (int)($result['token_count'] ?? 0),
            'wallet_count' => (int)($result['wallet_count'] ?? 0),
            'change_24h' => 2.5, // Mock data - implement real calculation
            'change_7d' => 5.8   // Mock data - implement real calculation
        ];
    }

    /**
     * Get farm summary
     */
    private function getFarmSummary($userId) {
        $query = "SELECT 
                    COUNT(*) as active_farms,
                    SUM(ufs.staked_amount) as total_staked,
                    SUM(ufs.pending_rewards) as total_rewards,
                    AVG(fp.apy) as avg_apy
                 FROM user_farm_stakes ufs 
                 JOIN farm_pools fp ON ufs.farm_pool_id = fp.id 
                 WHERE ufs.user_id = :user_id AND ufs.staked_amount > 0";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();

        $result = $stmt->fetch();
        
        return [
            'active_farms' => (int)($result['active_farms'] ?? 0),
            'total_staked' => (float)($result['total_staked'] ?? 0),
            'total_rewards' => (float)($result['total_rewards'] ?? 0),
            'avg_apy' => (float)($result['avg_apy'] ?? 0)
        ];
    }

    /**
     * Get staking summary
     */
    private function getStakingSummary($userId) {
        $query = "SELECT 
                    COUNT(*) as active_positions,
                    SUM(usp.staked_amount) as total_staked,
                    SUM(usp.pending_rewards) as total_rewards,
                    AVG(sp.apy) as avg_apy
                 FROM user_staking_positions usp 
                 JOIN staking_pools sp ON usp.staking_pool_id = sp.id 
                 WHERE usp.user_id = :user_id AND usp.staked_amount > 0";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();

        $result = $stmt->fetch();
        
        return [
            'active_positions' => (int)($result['active_positions'] ?? 0),
            'total_staked' => (float)($result['total_staked'] ?? 0),
            'total_rewards' => (float)($result['total_rewards'] ?? 0),
            'avg_apy' => (float)($result['avg_apy'] ?? 0)
        ];
    }

    /**
     * Get recent transactions
     */
    private function getRecentTransactions($userId, $limit = 10) {
        $query = "SELECT 
                    t.*,
                    tok.symbol as token_symbol,
                    tok.name as token_name,
                    sc.name as chain_name
                 FROM transactions t 
                 LEFT JOIN tokens tok ON t.token_id = tok.id 
                 JOIN supported_chains sc ON t.chain_id = sc.chain_id 
                 WHERE t.user_id = :user_id 
                 ORDER BY t.created_at DESC 
                 LIMIT :limit";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    /**
     * Get governance summary
     */
    private function getGovernanceSummary($userId) {
        // Get active proposals
        $query = "SELECT COUNT(*) as active_proposals 
                 FROM governance_proposals 
                 WHERE status = 'active' AND end_date > NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $activeProposals = $stmt->fetch()['active_proposals'];

        // Get user's voting power (simplified)
        $query = "SELECT SUM(usp.staked_amount) as voting_power 
                 FROM user_staking_positions usp 
                 WHERE usp.user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        $votingPower = $stmt->fetch()['voting_power'] ?? 0;

        // Get user's votes
        $query = "SELECT COUNT(*) as votes_cast 
                 FROM user_governance_votes ugv 
                 JOIN governance_proposals gp ON ugv.proposal_id = gp.id 
                 WHERE ugv.user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        $votesCast = $stmt->fetch()['votes_cast'] ?? 0;

        return [
            'active_proposals' => (int)$activeProposals,
            'voting_power' => (float)$votingPower,
            'votes_cast' => (int)$votesCast
        ];
    }

    /**
     * Get portfolio details
     */
    public function getPortfolio($userId) {
        try {
            $query = "SELECT 
                        utb.*,
                        t.symbol,
                        t.name,
                        t.decimals,
                        t.logo_url,
                        sc.name as chain_name,
                        uw.wallet_address
                     FROM user_token_balances utb 
                     JOIN tokens t ON utb.token_id = t.id 
                     JOIN supported_chains sc ON t.chain_id = sc.chain_id 
                     JOIN user_wallets uw ON utb.wallet_id = uw.id 
                     WHERE utb.user_id = :user_id AND utb.balance > 0
                     ORDER BY utb.balance DESC";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->execute();

            return $stmt->fetchAll();

        } catch (Exception $e) {
            error_log("Portfolio error: " . $e->getMessage());
            throw $e;
        }
    }
}
?>
