# FlareHub Backend API

A comprehensive PHP backend API for the FlareHub DeFi platform, providing endpoints for user management, wallet integration, farming, staking, governance, and more.

## 🚀 Features

- **User Authentication**: JWT-based authentication with secure session management
- **Wallet Integration**: Support for multiple wallet types (MetaMask, Coinbase, Trust Wallet, etc.)
- **DeFi Operations**: Farm staking, token staking, swapping, and bridging
- **Governance**: Proposal creation, voting, and result tracking
- **Portfolio Management**: Real-time balance tracking and transaction history
- **Rate Limiting**: Built-in API rate limiting and security measures
- **Database**: Comprehensive MySQL schema for all platform features

## 📁 Project Structure

```
backend/
├── config/
│   ├── config.php          # Application configuration
│   └── database.php        # Database connection settings
├── classes/
│   ├── Auth.php           # Authentication handler
│   ├── Dashboard.php      # Dashboard data management
│   ├── Farms.php          # Farm operations
│   ├── JWT.php            # JWT token handling
│   ├── RateLimiter.php    # API rate limiting
│   └── Response.php       # API response formatting
├── database/
│   └── schema.sql         # MySQL database schema
├── endpoints/
│   ├── auth/              # Authentication endpoints
│   ├── dashboard/         # Dashboard data endpoints
│   ├── farms/             # Farm operation endpoints
│   ├── staking/           # Staking endpoints
│   ├── swap/              # Token swap endpoints
│   ├── bridge/            # Cross-chain bridge endpoints
│   ├── governance/        # Governance endpoints
│   └── wallet/            # Wallet management endpoints
├── .htaccess              # Apache configuration
├── deploy.sh              # Deployment script
├── index.php              # Main API router
└── README.md              # This file
```

## 🛠️ Installation

### Prerequisites

- PHP 8.0 or higher
- MySQL 8.0 or higher
- Apache/Nginx web server
- SSL certificate (for production)

### 1. Clone and Setup

```bash
# Clone the repository
git clone https://github.com/yourusername/flarehub.git
cd flarehub/backend

# Set permissions
chmod +x deploy.sh
```

### 2. Configure Database

```bash
# Create database and user
mysql -u root -p
CREATE DATABASE flarehub CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'flarehub_user'@'localhost' IDENTIFIED BY 'your_secure_password';
GRANT ALL PRIVILEGES ON flarehub.* TO 'flarehub_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;

# Import schema
mysql -u flarehub_user -p flarehub < database/schema.sql
```

### 3. Configure Application

Edit `config/config.php` and `config/database.php` with your settings:

```php
// config/config.php
define('JWT_SECRET', 'your_jwt_secret_key_here');
define('ENCRYPTION_KEY', 'your_encryption_key_here');
define('ETHEREUM_RPC_URL', 'https://mainnet.infura.io/v3/your_key');

// config/database.php
private $username = 'flarehub_user';
private $password = 'your_secure_password';
```

### 4. Deploy

```bash
# Run deployment script
./deploy.sh
```

## 📚 API Endpoints

### Authentication
- `POST /api/v1/auth/register` - User registration
- `POST /api/v1/auth/login` - User login
- `POST /api/v1/auth/logout` - User logout
- `POST /api/v1/auth/wallet/connect` - Connect wallet

### Dashboard
- `GET /api/v1/dashboard/overview` - Dashboard overview data
- `GET /api/v1/dashboard/portfolio` - User portfolio details

### Farms
- `GET /api/v1/farms` - List farm pools
- `POST /api/v1/farms/stake` - Stake in farm pool
- `POST /api/v1/farms/unstake` - Unstake from farm pool
- `POST /api/v1/farms/harvest` - Harvest farm rewards

### Staking
- `GET /api/v1/staking/pools` - List staking pools
- `POST /api/v1/staking/stake` - Stake tokens
- `POST /api/v1/staking/unstake` - Unstake tokens
- `POST /api/v1/staking/claim` - Claim staking rewards

### Swap
- `POST /api/v1/swap/quote` - Get swap quote
- `POST /api/v1/swap/execute` - Execute token swap

### Bridge
- `POST /api/v1/bridge/estimate` - Get bridge estimate
- `POST /api/v1/bridge/execute` - Execute cross-chain bridge

### Governance
- `GET /api/v1/governance/proposals` - List governance proposals
- `POST /api/v1/governance/vote` - Vote on proposal

### Wallet
- `GET /api/v1/wallet/balance` - Get wallet balances
- `GET /api/v1/wallet/transactions` - Get transaction history

### System
- `GET /api/v1/health` - Health check
- `GET /api/v1/docs` - API documentation

## 🔐 Security Features

- **JWT Authentication**: Secure token-based authentication
- **Rate Limiting**: Prevents API abuse with configurable limits
- **CORS Protection**: Configurable cross-origin resource sharing
- **Input Validation**: Comprehensive input sanitization and validation
- **SQL Injection Protection**: Prepared statements for all database queries
- **XSS Protection**: Output encoding and security headers
- **HTTPS Enforcement**: SSL/TLS encryption for all communications

## 🗄️ Database Schema

The database includes tables for:

- **Users**: User accounts and profiles
- **User Wallets**: Connected wallet addresses
- **Supported Chains**: Blockchain network configurations
- **Tokens**: Token information and metadata
- **User Token Balances**: Real-time balance tracking
- **Farm Pools**: Liquidity farming pools
- **User Farm Stakes**: User farm positions
- **Staking Pools**: Token staking pools
- **User Staking Positions**: User staking positions
- **Transactions**: Transaction history and tracking
- **Governance Proposals**: Governance proposals and voting
- **User Governance Votes**: User voting records
- **API Rate Limits**: Rate limiting data
- **User Sessions**: Active user sessions

## 🔧 Configuration

### Environment Variables

Set these in your server environment or config files:

```bash
# Database
DB_HOST=localhost
DB_NAME=flarehub
DB_USER=flarehub_user
DB_PASS=your_secure_password

# JWT
JWT_SECRET=your_jwt_secret_key
JWT_EXPIRY=604800

# API Keys
INFURA_API_KEY=your_infura_key
COINGECKO_API_KEY=your_coingecko_key
ETHERSCAN_API_KEY=your_etherscan_key

# Security
ENCRYPTION_KEY=your_encryption_key
RATE_LIMIT_REQUESTS=100
RATE_LIMIT_WINDOW=3600
```

### CORS Configuration

Update `ALLOWED_ORIGINS` in `config/config.php`:

```php
define('ALLOWED_ORIGINS', [
    'https://flarehub.net',
    'https://www.flarehub.net',
    'https://flarehub-mu.vercel.app'
]);
```

## 📊 Monitoring

### Log Files

- Application logs: `/var/www/flarehub.net/api/logs/`
- Apache logs: `/var/log/apache2/flarehub_*.log`
- Error logs: `/var/log/apache2/flarehub_error.log`

### Health Checks

Monitor these endpoints:
- `GET /api/v1/health` - API health status
- Database connectivity
- External API availability

## 🚀 Deployment

### Production Deployment

1. **Server Setup**:
   ```bash
   # Update system
   sudo apt update && sudo apt upgrade -y
   
   # Install required packages
   sudo apt install -y apache2 mysql-server php8.1 php8.1-mysql php8.1-curl php8.1-json
   ```

2. **SSL Certificate**:
   ```bash
   # Install Certbot
   sudo apt install -y certbot python3-certbot-apache
   
   # Get SSL certificate
   sudo certbot --apache -d flarehub.net -d www.flarehub.net
   ```

3. **Database Setup**:
   ```bash
   # Secure MySQL installation
   sudo mysql_secure_installation
   
   # Create database and user
   mysql -u root -p < database/schema.sql
   ```

4. **Deploy Application**:
   ```bash
   # Run deployment script
   ./deploy.sh
   ```

### Docker Deployment (Alternative)

```dockerfile
FROM php:8.1-apache

# Install extensions
RUN docker-php-ext-install pdo pdo_mysql

# Copy application
COPY . /var/www/html/

# Set permissions
RUN chown -R www-data:www-data /var/www/html
RUN chmod -R 755 /var/www/html

# Enable Apache modules
RUN a2enmod rewrite headers ssl

EXPOSE 80 443
```

## 🔄 Updates

To update the backend:

1. **Backup Database**:
   ```bash
   mysqldump -u flarehub_user -p flarehub > backup_$(date +%Y%m%d).sql
   ```

2. **Update Code**:
   ```bash
   git pull origin main
   ```

3. **Run Migrations** (if any):
   ```bash
   mysql -u flarehub_user -p flarehub < database/migrations.sql
   ```

4. **Restart Services**:
   ```bash
   sudo systemctl restart apache2
   ```

## 🐛 Troubleshooting

### Common Issues

1. **Database Connection Error**:
   - Check database credentials in `config/database.php`
   - Verify MySQL service is running
   - Check firewall settings

2. **JWT Token Issues**:
   - Verify JWT_SECRET is set correctly
   - Check token expiration settings
   - Ensure proper Authorization header format

3. **CORS Errors**:
   - Update ALLOWED_ORIGINS in config
   - Check .htaccess configuration
   - Verify Apache headers module is enabled

4. **Rate Limiting Issues**:
   - Check rate limit settings in config
   - Clear rate limit table if needed
   - Verify IP address detection

### Debug Mode

Enable debug mode in `config/config.php`:

```php
define('ENVIRONMENT', 'development');
```

## 📞 Support

For support and questions:
- Email: support@flarehub.net
- Documentation: https://flarehub.net/docs
- GitHub Issues: https://github.com/yourusername/flarehub/issues

## 📄 License

This project is licensed under the MIT License - see the LICENSE file for details.

---

**FlareHub Backend API** - Powering the future of decentralized finance.
