<?php
/**
 * Update User Profile Endpoint
 * PUT /api/v1/user/update-profile
 */

require_once __DIR__ . '/../../classes/Auth.php';

// Get token from Authorization header
$token = JWT::extractToken();

if (!$token) {
    Response::unauthorized('Token required');
}

// Verify token
$payload = JWT::verify($token);
if (!$payload) {
    Response::unauthorized('Invalid token');
}

$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!$input) {
    Response::validationError(['data' => 'Profile data is required']);
}

$email = $input['email'] ?? null;
$username = $input['username'] ?? null;
$firstName = $input['first_name'] ?? null;
$lastName = $input['last_name'] ?? null;
$bio = $input['bio'] ?? null;
$avatarUrl = $input['avatar_url'] ?? null;

// Validate email if provided
if ($email && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    Response::validationError(['email' => 'Invalid email format']);
}

// Validate username if provided
if ($username && !preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
    Response::validationError(['username' => 'Username must be 3-20 characters, alphanumeric and underscore only']);
}

// Validate names if provided
if ($firstName && strlen($firstName) > 100) {
    Response::validationError(['first_name' => 'First name must be less than 100 characters']);
}

if ($lastName && strlen($lastName) > 100) {
    Response::validationError(['last_name' => 'Last name must be less than 100 characters']);
}

// Validate bio if provided
if ($bio && strlen($bio) > 500) {
    Response::validationError(['bio' => 'Bio must be less than 500 characters']);
}

try {
    $auth = new Auth();
    $result = $auth->updateUserProfile($payload['user_id'], [
        'email' => $email,
        'username' => $username,
        'first_name' => $firstName,
        'last_name' => $lastName,
        'bio' => $bio,
        'avatar_url' => $avatarUrl
    ]);

    if ($result['success']) {
        Response::success($result['user'], 'Profile updated successfully');
    } else {
        Response::error($result['message'], 400);
    }

} catch (Exception $e) {
    error_log("Profile update error: " . $e->getMessage());
    Response::internalError('Failed to update profile');
}
?>
