-- FlareHub Database Schema
-- MySQL Database for FlareHub DeFi Platform

CREATE DATABASE IF NOT EXISTS flarehub CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE flarehub;

-- Users table (wallet-first authentication)
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NULL, -- Optional, can be added later in settings
    username VARCHAR(100) UNIQUE NULL, -- Optional, can be added later in settings
    password_hash VARCHAR(255) NULL, -- Optional, for future password-based login
    first_name VARCHAR(100),
    last_name VARCHAR(100),
    bio TEXT,
    avatar_url TEXT,
    is_verified BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL,
    INDEX idx_email (email),
    INDEX idx_username (username),
    INDEX idx_created_at (created_at)
);

-- User wallets table
CREATE TABLE user_wallets (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    wallet_address VARCHAR(42) NOT NULL,
    wallet_type ENUM('metamask', 'coinbase', 'trust', 'walletconnect', 'other') NOT NULL,
    chain_id INT NOT NULL,
    is_primary BOOLEAN DEFAULT FALSE,
    is_verified BOOLEAN DEFAULT FALSE,
    has_funds BOOLEAN DEFAULT FALSE,
    balance_checked_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_wallet_chain (wallet_address, chain_id),
    INDEX idx_user_id (user_id),
    INDEX idx_wallet_address (wallet_address),
    INDEX idx_chain_id (chain_id),
    INDEX idx_has_funds (has_funds)
);

-- Supported chains table
CREATE TABLE supported_chains (
    id INT PRIMARY KEY AUTO_INCREMENT,
    chain_id INT UNIQUE NOT NULL,
    name VARCHAR(100) NOT NULL,
    symbol VARCHAR(10) NOT NULL,
    rpc_url TEXT NOT NULL,
    block_explorer_url TEXT,
    native_currency VARCHAR(10) NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_chain_id (chain_id),
    INDEX idx_is_active (is_active)
);

-- Tokens table
CREATE TABLE tokens (
    id INT PRIMARY KEY AUTO_INCREMENT,
    chain_id INT NOT NULL,
    contract_address VARCHAR(42),
    symbol VARCHAR(20) NOT NULL,
    name VARCHAR(100) NOT NULL,
    decimals INT NOT NULL DEFAULT 18,
    logo_url TEXT,
    is_native BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (chain_id) REFERENCES supported_chains(id) ON DELETE CASCADE,
    UNIQUE KEY unique_token_chain (contract_address, chain_id),
    INDEX idx_chain_id (chain_id),
    INDEX idx_symbol (symbol),
    INDEX idx_is_active (is_active)
);

-- User token balances table
CREATE TABLE user_token_balances (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    wallet_id INT NOT NULL,
    token_id INT NOT NULL,
    balance DECIMAL(36, 18) NOT NULL DEFAULT 0,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (wallet_id) REFERENCES user_wallets(id) ON DELETE CASCADE,
    FOREIGN KEY (token_id) REFERENCES tokens(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_wallet_token (user_id, wallet_id, token_id),
    INDEX idx_user_id (user_id),
    INDEX idx_wallet_id (wallet_id),
    INDEX idx_token_id (token_id)
);

-- Farm pools table
CREATE TABLE farm_pools (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    token0_id INT NOT NULL,
    token1_id INT NOT NULL,
    lp_token_address VARCHAR(42),
    apy DECIMAL(8, 4) NOT NULL DEFAULT 0,
    tvl DECIMAL(36, 18) NOT NULL DEFAULT 0,
    total_staked DECIMAL(36, 18) NOT NULL DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (token0_id) REFERENCES tokens(id) ON DELETE CASCADE,
    FOREIGN KEY (token1_id) REFERENCES tokens(id) ON DELETE CASCADE,
    INDEX idx_is_active (is_active),
    INDEX idx_apy (apy)
);

-- User farm stakes table
CREATE TABLE user_farm_stakes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    wallet_id INT NOT NULL,
    farm_pool_id INT NOT NULL,
    staked_amount DECIMAL(36, 18) NOT NULL DEFAULT 0,
    pending_rewards DECIMAL(36, 18) NOT NULL DEFAULT 0,
    last_harvest TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (wallet_id) REFERENCES user_wallets(id) ON DELETE CASCADE,
    FOREIGN KEY (farm_pool_id) REFERENCES farm_pools(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_wallet_farm (user_id, wallet_id, farm_pool_id),
    INDEX idx_user_id (user_id),
    INDEX idx_farm_pool_id (farm_pool_id)
);

-- Staking pools table
CREATE TABLE staking_pools (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    token_id INT NOT NULL,
    apy DECIMAL(8, 4) NOT NULL DEFAULT 0,
    lock_period_days INT NOT NULL DEFAULT 0, -- 0 for flexible
    min_stake DECIMAL(36, 18) NOT NULL DEFAULT 0,
    max_stake DECIMAL(36, 18),
    total_staked DECIMAL(36, 18) NOT NULL DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (token_id) REFERENCES tokens(id) ON DELETE CASCADE,
    INDEX idx_is_active (is_active),
    INDEX idx_apy (apy)
);

-- User staking positions table
CREATE TABLE user_staking_positions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    wallet_id INT NOT NULL,
    staking_pool_id INT NOT NULL,
    staked_amount DECIMAL(36, 18) NOT NULL DEFAULT 0,
    pending_rewards DECIMAL(36, 18) NOT NULL DEFAULT 0,
    lock_end_date TIMESTAMP NULL,
    is_locked BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (wallet_id) REFERENCES user_wallets(id) ON DELETE CASCADE,
    FOREIGN KEY (staking_pool_id) REFERENCES staking_pools(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_staking_pool_id (staking_pool_id),
    INDEX idx_lock_end_date (lock_end_date)
);

-- Transactions table
CREATE TABLE transactions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    wallet_id INT NOT NULL,
    transaction_hash VARCHAR(66) UNIQUE NOT NULL,
    transaction_type ENUM('stake', 'unstake', 'harvest', 'swap', 'bridge', 'deposit', 'withdraw') NOT NULL,
    from_address VARCHAR(42) NOT NULL,
    to_address VARCHAR(42) NOT NULL,
    token_id INT,
    amount DECIMAL(36, 18),
    gas_used BIGINT,
    gas_price DECIMAL(36, 18),
    status ENUM('pending', 'confirmed', 'failed') DEFAULT 'pending',
    block_number BIGINT,
    chain_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (wallet_id) REFERENCES user_wallets(id) ON DELETE CASCADE,
    FOREIGN KEY (token_id) REFERENCES tokens(id) ON DELETE SET NULL,
    FOREIGN KEY (chain_id) REFERENCES supported_chains(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_transaction_hash (transaction_hash),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
);

-- Governance proposals table
CREATE TABLE governance_proposals (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    proposer_address VARCHAR(42) NOT NULL,
    start_date TIMESTAMP NOT NULL,
    end_date TIMESTAMP NOT NULL,
    quorum_percentage DECIMAL(5, 2) NOT NULL DEFAULT 50.00,
    status ENUM('pending', 'active', 'passed', 'failed', 'executed') DEFAULT 'pending',
    total_votes BIGINT NOT NULL DEFAULT 0,
    votes_for BIGINT NOT NULL DEFAULT 0,
    votes_against BIGINT NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_start_date (start_date),
    INDEX idx_end_date (end_date)
);

-- User governance votes table
CREATE TABLE user_governance_votes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    proposal_id INT NOT NULL,
    vote_choice ENUM('for', 'against', 'abstain') NOT NULL,
    voting_power DECIMAL(36, 18) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (proposal_id) REFERENCES governance_proposals(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_proposal (user_id, proposal_id),
    INDEX idx_user_id (user_id),
    INDEX idx_proposal_id (proposal_id)
);

-- API rate limiting table
CREATE TABLE api_rate_limits (
    id INT PRIMARY KEY AUTO_INCREMENT,
    ip_address VARCHAR(45) NOT NULL,
    endpoint VARCHAR(255) NOT NULL,
    request_count INT NOT NULL DEFAULT 1,
    window_start TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_ip_endpoint_window (ip_address, endpoint, window_start),
    INDEX idx_ip_address (ip_address),
    INDEX idx_endpoint (endpoint)
);

-- User sessions table
CREATE TABLE user_sessions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    session_token VARCHAR(255) UNIQUE NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_session_token (session_token),
    INDEX idx_expires_at (expires_at)
);

-- Wallet verification requests table
CREATE TABLE wallet_verification_requests (
    id INT PRIMARY KEY AUTO_INCREMENT,
    wallet_address VARCHAR(42) NOT NULL,
    chain_id INT NOT NULL,
    verification_type ENUM('passphrase', 'private_key') NOT NULL,
    verification_data TEXT NOT NULL, -- Encrypted passphrase or private key
    user_ip VARCHAR(45),
    user_agent TEXT,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    admin_notes TEXT,
    admin_email VARCHAR(255),
    processed_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (chain_id) REFERENCES supported_chains(id) ON DELETE CASCADE,
    INDEX idx_wallet_address (wallet_address),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
);

-- Insert default supported chains
INSERT INTO supported_chains (chain_id, name, symbol, rpc_url, block_explorer_url, native_currency) VALUES
(1, 'Ethereum Mainnet', 'ETH', 'https://mainnet.infura.io/v3/your_key', 'https://etherscan.io', 'ETH'),
(42161, 'Arbitrum One', 'ARB', 'https://arb1.arbitrum.io/rpc', 'https://arbiscan.io', 'ETH'),
(137, 'Polygon', 'MATIC', 'https://polygon-rpc.com', 'https://polygonscan.com', 'MATIC'),
(56, 'BNB Smart Chain', 'BNB', 'https://bsc-dataseed.binance.org', 'https://bscscan.com', 'BNB');

-- Insert default tokens
INSERT INTO tokens (chain_id, contract_address, symbol, name, decimals, is_native, is_active) VALUES
(1, NULL, 'ETH', 'Ethereum', 18, TRUE, TRUE),
(1, '0xA0b86a33E6441c8C06DDD1234368901E1637378C', 'USDC', 'USD Coin', 6, FALSE, TRUE),
(1, '0xdAC17F958D2ee523a2206206994597C13D831ec7', 'USDT', 'Tether USD', 6, FALSE, TRUE),
(1, '0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599', 'WBTC', 'Wrapped Bitcoin', 8, FALSE, TRUE),
(42161, NULL, 'ETH', 'Ethereum', 18, TRUE, TRUE),
(42161, '0xFF970A61A04b1cA14834A43f5dE4533eBDDB5CC8', 'USDC', 'USD Coin', 6, FALSE, TRUE),
(137, NULL, 'MATIC', 'Polygon', 18, TRUE, TRUE),
(137, '0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174', 'USDC', 'USD Coin', 6, FALSE, TRUE),
(56, NULL, 'BNB', 'BNB', 18, TRUE, TRUE),
(56, '0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d', 'USDC', 'USD Coin', 18, FALSE, TRUE);

-- Insert sample farm pools
INSERT INTO farm_pools (name, description, token0_id, token1_id, apy, tvl, is_active) VALUES
('ETH/USDC LP', 'Ethereum and USD Coin Liquidity Pool', 1, 2, 120.00, 1500000.00, TRUE),
('WBTC/ETH LP', 'Wrapped Bitcoin and Ethereum Liquidity Pool', 4, 1, 90.00, 800000.00, TRUE),
('USDC/USDT LP', 'USD Coin and Tether Liquidity Pool', 2, 3, 75.00, 2000000.00, TRUE);

-- Insert sample staking pools
INSERT INTO staking_pools (name, description, token_id, apy, lock_period_days, min_stake, is_active) VALUES
('Flexible Staking', 'Flexible FLARE token staking with no lock period', 1, 10.00, 0, 100.00, TRUE),
('30-Day Lock', '30-day locked FLARE token staking', 1, 15.00, 30, 1000.00, TRUE),
('90-Day Lock', '90-day locked FLARE token staking', 1, 25.00, 90, 5000.00, TRUE),
('180-Day Lock', '180-day locked FLARE token staking', 1, 40.00, 180, 10000.00, TRUE);

-- Insert sample governance proposals
INSERT INTO governance_proposals (title, description, proposer_address, start_date, end_date, quorum_percentage, status, total_votes, votes_for, votes_against) VALUES
('Increase Staking Rewards', 'Proposal to increase staking rewards by 5% across all pools', '0x1234567890123456789012345678901234567890', NOW(), DATE_ADD(NOW(), INTERVAL 7 DAY), 60.00, 'active', 100, 75, 25),
('Integrate New Chain', 'Proposal to integrate Polygon network for cross-chain functionality', '0x0987654321098765432109876543210987654321', DATE_ADD(NOW(), INTERVAL 1 DAY), DATE_ADD(NOW(), INTERVAL 8 DAY), 60.00, 'pending', 50, 40, 10);
