<?php
/**
 * Rate Limiter
 * FlareHub Backend API
 */

require_once __DIR__ . '/../config/database.php';

class RateLimiter {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Check if request is within rate limit
     */
    public function checkLimit($ipAddress, $endpoint) {
        try {
            $windowStart = date('Y-m-d H:i:s', time() - RATE_LIMIT_WINDOW);
            
            // Clean old entries
            $this->cleanOldEntries($windowStart);
            
            // Check current count
            $query = "SELECT request_count FROM api_rate_limits 
                     WHERE ip_address = :ip_address 
                     AND endpoint = :endpoint 
                     AND window_start >= :window_start";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':ip_address', $ipAddress);
            $stmt->bindParam(':endpoint', $endpoint);
            $stmt->bindParam(':window_start', $windowStart);
            $stmt->execute();
            
            $result = $stmt->fetch();
            
            if ($result) {
                if ($result['request_count'] >= RATE_LIMIT_REQUESTS) {
                    return false;
                }
                
                // Increment count
                $query = "UPDATE api_rate_limits 
                         SET request_count = request_count + 1 
                         WHERE ip_address = :ip_address 
                         AND endpoint = :endpoint 
                         AND window_start >= :window_start";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':ip_address', $ipAddress);
                $stmt->bindParam(':endpoint', $endpoint);
                $stmt->bindParam(':window_start', $windowStart);
                $stmt->execute();
            } else {
                // Create new entry
                $query = "INSERT INTO api_rate_limits (ip_address, endpoint, request_count, window_start) 
                         VALUES (:ip_address, :endpoint, 1, NOW())";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':ip_address', $ipAddress);
                $stmt->bindParam(':endpoint', $endpoint);
                $stmt->execute();
            }
            
            return true;
            
        } catch (Exception $e) {
            error_log("Rate limiter error: " . $e->getMessage());
            return true; // Allow request on error
        }
    }

    /**
     * Clean old rate limit entries
     */
    private function cleanOldEntries($windowStart) {
        $query = "DELETE FROM api_rate_limits WHERE window_start < :window_start";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':window_start', $windowStart);
        $stmt->execute();
    }

    /**
     * Get remaining requests for IP and endpoint
     */
    public function getRemainingRequests($ipAddress, $endpoint) {
        try {
            $windowStart = date('Y-m-d H:i:s', time() - RATE_LIMIT_WINDOW);
            
            $query = "SELECT request_count FROM api_rate_limits 
                     WHERE ip_address = :ip_address 
                     AND endpoint = :endpoint 
                     AND window_start >= :window_start";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':ip_address', $ipAddress);
            $stmt->bindParam(':endpoint', $endpoint);
            $stmt->bindParam(':window_start', $windowStart);
            $stmt->execute();
            
            $result = $stmt->fetch();
            $used = $result ? $result['request_count'] : 0;
            
            return max(0, RATE_LIMIT_REQUESTS - $used);
            
        } catch (Exception $e) {
            error_log("Get remaining requests error: " . $e->getMessage());
            return RATE_LIMIT_REQUESTS;
        }
    }
}
?>
