<?php
/**
 * Farms Handler
 * FlareHub Backend API
 */

require_once __DIR__ . '/../config/database.php';

class Farms {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Get farm pools with user stakes
     */
    public function getFarmPools($userId) {
        try {
            $query = "SELECT 
                        fp.*,
                        t0.symbol as token0_symbol,
                        t0.name as token0_name,
                        t1.symbol as token1_symbol,
                        t1.name as token1_name,
                        ufs.staked_amount,
                        ufs.pending_rewards,
                        ufs.last_harvest
                     FROM farm_pools fp 
                     JOIN tokens t0 ON fp.token0_id = t0.id 
                     JOIN tokens t1 ON fp.token1_id = t1.id 
                     LEFT JOIN user_farm_stakes ufs ON fp.id = ufs.farm_pool_id AND ufs.user_id = :user_id 
                     WHERE fp.is_active = TRUE 
                     ORDER BY fp.apy DESC";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->execute();

            $farms = $stmt->fetchAll();

            // Format the response
            foreach ($farms as &$farm) {
                $farm['staked_amount'] = (float)($farm['staked_amount'] ?? 0);
                $farm['pending_rewards'] = (float)($farm['pending_rewards'] ?? 0);
                $farm['apy'] = (float)$farm['apy'];
                $farm['tvl'] = (float)$farm['tvl'];
                $farm['total_staked'] = (float)$farm['total_staked'];
            }

            return $farms;

        } catch (Exception $e) {
            error_log("Get farm pools error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Stake in farm pool
     */
    public function stake($userId, $walletId, $farmPoolId, $amount) {
        try {
            $this->conn->beginTransaction();

            // Check if user already has a stake in this pool
            $query = "SELECT id, staked_amount FROM user_farm_stakes 
                     WHERE user_id = :user_id AND farm_pool_id = :farm_pool_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':farm_pool_id', $farmPoolId);
            $stmt->execute();

            $existingStake = $stmt->fetch();

            if ($existingStake) {
                // Update existing stake
                $newAmount = $existingStake['staked_amount'] + $amount;
                $query = "UPDATE user_farm_stakes 
                         SET staked_amount = :staked_amount, updated_at = CURRENT_TIMESTAMP 
                         WHERE id = :id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':staked_amount', $newAmount);
                $stmt->bindParam(':id', $existingStake['id']);
                $stmt->execute();
            } else {
                // Create new stake
                $query = "INSERT INTO user_farm_stakes (user_id, wallet_id, farm_pool_id, staked_amount) 
                         VALUES (:user_id, :wallet_id, :farm_pool_id, :staked_amount)";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':user_id', $userId);
                $stmt->bindParam(':wallet_id', $walletId);
                $stmt->bindParam(':farm_pool_id', $farmPoolId);
                $stmt->bindParam(':staked_amount', $amount);
                $stmt->execute();
            }

            // Update farm pool total staked
            $query = "UPDATE farm_pools 
                     SET total_staked = total_staked + :amount, updated_at = CURRENT_TIMESTAMP 
                     WHERE id = :farm_pool_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':amount', $amount);
            $stmt->bindParam(':farm_pool_id', $farmPoolId);
            $stmt->execute();

            // Record transaction
            $this->recordTransaction($userId, $walletId, 'stake', $farmPoolId, $amount);

            $this->conn->commit();

            return [
                'success' => true,
                'message' => 'Successfully staked in farm pool',
                'staked_amount' => $amount
            ];

        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Farm stake error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Unstake from farm pool
     */
    public function unstake($userId, $walletId, $farmPoolId, $amount) {
        try {
            $this->conn->beginTransaction();

            // Check user's stake
            $query = "SELECT id, staked_amount FROM user_farm_stakes 
                     WHERE user_id = :user_id AND farm_pool_id = :farm_pool_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':farm_pool_id', $farmPoolId);
            $stmt->execute();

            $stake = $stmt->fetch();

            if (!$stake || $stake['staked_amount'] < $amount) {
                throw new Exception('Insufficient staked amount');
            }

            $newAmount = $stake['staked_amount'] - $amount;

            if ($newAmount <= 0) {
                // Remove stake completely
                $query = "DELETE FROM user_farm_stakes WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':id', $stake['id']);
                $stmt->execute();
            } else {
                // Update stake
                $query = "UPDATE user_farm_stakes 
                         SET staked_amount = :staked_amount, updated_at = CURRENT_TIMESTAMP 
                         WHERE id = :id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':staked_amount', $newAmount);
                $stmt->bindParam(':id', $stake['id']);
                $stmt->execute();
            }

            // Update farm pool total staked
            $query = "UPDATE farm_pools 
                     SET total_staked = total_staked - :amount, updated_at = CURRENT_TIMESTAMP 
                     WHERE id = :farm_pool_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':amount', $amount);
            $stmt->bindParam(':farm_pool_id', $farmPoolId);
            $stmt->execute();

            // Record transaction
            $this->recordTransaction($userId, $walletId, 'unstake', $farmPoolId, $amount);

            $this->conn->commit();

            return [
                'success' => true,
                'message' => 'Successfully unstaked from farm pool',
                'unstaked_amount' => $amount
            ];

        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Farm unstake error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Harvest rewards from farm pool
     */
    public function harvest($userId, $walletId, $farmPoolId) {
        try {
            $this->conn->beginTransaction();

            // Get user's stake and pending rewards
            $query = "SELECT id, pending_rewards FROM user_farm_stakes 
                     WHERE user_id = :user_id AND farm_pool_id = :farm_pool_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':farm_pool_id', $farmPoolId);
            $stmt->execute();

            $stake = $stmt->fetch();

            if (!$stake || $stake['pending_rewards'] <= 0) {
                throw new Exception('No rewards to harvest');
            }

            $rewards = $stake['pending_rewards'];

            // Reset pending rewards
            $query = "UPDATE user_farm_stakes 
                     SET pending_rewards = 0, last_harvest = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP 
                     WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $stake['id']);
            $stmt->execute();

            // Record transaction
            $this->recordTransaction($userId, $walletId, 'harvest', $farmPoolId, $rewards);

            $this->conn->commit();

            return [
                'success' => true,
                'message' => 'Successfully harvested rewards',
                'harvested_amount' => $rewards
            ];

        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Farm harvest error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Record transaction
     */
    private function recordTransaction($userId, $walletId, $type, $farmPoolId, $amount) {
        // Get wallet address and chain info
        $query = "SELECT uw.wallet_address, uw.chain_id 
                 FROM user_wallets uw 
                 WHERE uw.id = :wallet_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':wallet_id', $walletId);
        $stmt->execute();

        $wallet = $stmt->fetch();

        // Insert transaction record
        $query = "INSERT INTO transactions (user_id, wallet_id, transaction_type, from_address, to_address, amount, chain_id, status) 
                 VALUES (:user_id, :wallet_id, :transaction_type, :from_address, :to_address, :amount, :chain_id, 'pending')";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':wallet_id', $walletId);
        $stmt->bindParam(':transaction_type', $type);
        $stmt->bindParam(':from_address', $wallet['wallet_address']);
        $stmt->bindParam(':to_address', '0x0000000000000000000000000000000000000000'); // Contract address
        $stmt->bindParam(':amount', $amount);
        $stmt->bindParam(':chain_id', $wallet['chain_id']);
        $stmt->execute();
    }
}
?>
