<?php
/**
 * Etherscan Multichain API Handler
 * FlareHub Backend API
 */

class EtherscanAPI {
    private $apiKey;
    private $baseUrl;
    private $supportedChains;
    private $fallbackRpcUrls;

    public function __construct() {
        $this->apiKey = ETHERSCAN_API_KEY;
        $this->baseUrl = ETHERSCAN_API_BASE_URL;
        $this->supportedChains = SUPPORTED_CHAINS;
        $this->fallbackRpcUrls = FALLBACK_RPC_URLS;
    }

    /**
     * Get wallet balance across multiple chains
     */
    public function getWalletBalance($address, $chainId = 1) {
        if (!isset($this->supportedChains[$chainId])) {
            return [
                'success' => false,
                'message' => 'Unsupported chain ID: ' . $chainId
            ];
        }

        $url = $this->baseUrl . '?' . http_build_query([
            'chainid' => $chainId,
            'action' => 'balance',
            'address' => $address,
            'apikey' => $this->apiKey
        ]);

        $response = $this->makeRequest($url);
        
        if ($response['success']) {
            $balance = $response['data']['result'] ?? '0';
            return [
                'success' => true,
                'chain_id' => $chainId,
                'chain_name' => $this->supportedChains[$chainId],
                'address' => $address,
                'balance' => $balance,
                'balance_eth' => $this->weiToEth($balance)
            ];
        }

        return $response;
    }

    /**
     * Get transaction history for a wallet
     */
    public function getTransactionHistory($address, $chainId = 1, $startBlock = 0, $endBlock = 99999999) {
        if (!isset($this->supportedChains[$chainId])) {
            return [
                'success' => false,
                'message' => 'Unsupported chain ID: ' . $chainId
            ];
        }

        $url = $this->baseUrl . '?' . http_build_query([
            'chainid' => $chainId,
            'action' => 'txlist',
            'address' => $address,
            'startblock' => $startBlock,
            'endblock' => $endBlock,
            'sort' => 'desc',
            'apikey' => $this->apiKey
        ]);

        return $this->makeRequest($url);
    }

    /**
     * Get token balance for ERC-20 tokens
     */
    public function getTokenBalance($address, $contractAddress, $chainId = 1) {
        if (!isset($this->supportedChains[$chainId])) {
            return [
                'success' => false,
                'message' => 'Unsupported chain ID: ' . $chainId
            ];
        }

        $url = $this->baseUrl . '?' . http_build_query([
            'chainid' => $chainId,
            'action' => 'tokenbalance',
            'contractaddress' => $contractAddress,
            'address' => $address,
            'apikey' => $this->apiKey
        ]);

        return $this->makeRequest($url);
    }

    /**
     * Get token list for a wallet
     */
    public function getTokenList($address, $chainId = 1) {
        if (!isset($this->supportedChains[$chainId])) {
            return [
                'success' => false,
                'message' => 'Unsupported chain ID: ' . $chainId
            ];
        }

        $url = $this->baseUrl . '?' . http_build_query([
            'chainid' => $chainId,
            'action' => 'tokenlist',
            'address' => $address,
            'apikey' => $this->apiKey
        ]);

        return $this->makeRequest($url);
    }

    /**
     * Get gas price for a chain
     */
    public function getGasPrice($chainId = 1) {
        if (!isset($this->supportedChains[$chainId])) {
            return [
                'success' => false,
                'message' => 'Unsupported chain ID: ' . $chainId
            ];
        }

        $url = $this->baseUrl . '?' . http_build_query([
            'chainid' => $chainId,
            'action' => 'gasprice',
            'apikey' => $this->apiKey
        ]);

        return $this->makeRequest($url);
    }

    /**
     * Check if wallet has any funds across multiple chains
     */
    public function checkWalletFunds($address, $chainIds = [1, 137, 56, 42161]) {
        $results = [];
        $hasFunds = false;
        $totalBalance = 0;

        foreach ($chainIds as $chainId) {
            $balance = $this->getWalletBalance($address, $chainId);
            
            if ($balance['success']) {
                $balanceWei = $balance['balance'];
                $balanceEth = $balance['balance_eth'];
                
                $results[$chainId] = [
                    'chain_id' => $chainId,
                    'chain_name' => $balance['chain_name'],
                    'balance' => $balanceWei,
                    'balance_eth' => $balanceEth,
                    'has_funds' => $balanceWei > 0
                ];

                if ($balanceWei > 0) {
                    $hasFunds = true;
                    $totalBalance += $balanceEth;
                }
            }
        }

        return [
            'success' => true,
            'address' => $address,
            'has_funds' => $hasFunds,
            'total_balance_eth' => $totalBalance,
            'chain_balances' => $results
        ];
    }

    /**
     * Get supported chains
     */
    public function getSupportedChains() {
        return $this->supportedChains;
    }

    /**
     * Make HTTP request to Etherscan API
     */
    private function makeRequest($url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_USERAGENT, 'FlareHub API/1.0');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return [
                'success' => false,
                'message' => 'CURL Error: ' . $error
            ];
        }

        if ($httpCode !== 200) {
            return [
                'success' => false,
                'message' => 'HTTP Error: ' . $httpCode
            ];
        }

        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => false,
                'message' => 'JSON Decode Error: ' . json_last_error_msg()
            ];
        }

        // Check for Etherscan API errors
        if (isset($data['status']) && $data['status'] === '0' && isset($data['message'])) {
            return [
                'success' => false,
                'message' => 'Etherscan API Error: ' . $data['message']
            ];
        }

        return [
            'success' => true,
            'data' => $data
        ];
    }

    /**
     * Convert Wei to ETH
     */
    private function weiToEth($wei) {
        return bcdiv($wei, '1000000000000000000', 18);
    }

    /**
     * Get fallback RPC URL for a chain
     */
    public function getFallbackRpcUrl($chainId) {
        return $this->fallbackRpcUrls[$chainId] ?? null;
    }
}
?>
